<?php
// trys.php — SD/DP live viewer with tabs, notifications & sound
ini_set('display_errors', 0);
error_reporting(E_ALL);
session_start();
date_default_timezone_set('Asia/Kolkata');

$BASE = __DIR__;
$conn = null;
$DB_FILE = $BASE . '/db.php';
if (is_file($DB_FILE)) { require $DB_FILE; if (!isset($conn) || !($conn instanceof mysqli)) $conn = null; }

// ---- AJAX endpoint: return cached rows ----
if (isset($_GET['ajax'])) {
  $type = ($_GET['type'] ?? 'sd') === 'dp' ? 'dp' : 'sd';
  header('Content-Type: text/html; charset=utf-8');
  header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');

  $file = $BASE . '/cache/user_bids_rows_' . $type . '.html';
  if (is_file($file)) { readfile($file); exit; }

  // Fallback to DB if cache not ready (no crash if db.php missing)
  if ($conn) {
    $table   = ($type==='dp') ? 'user_bids_dp' : 'user_bids';
    $openCol = ($type==='dp') ? 'winning_open_time_dp'  : 'winning_open_time_sd';
    $closeCol= ($type==='dp') ? 'winning_close_time_dp' : 'winning_close_time_sd';
    $sql = "SELECT id,user_id,game_id,bid_details,total_bid_amount,created_at,
                   game_name,bid_session,{$openCol} AS w_open, {$closeCol} AS w_close
            FROM {$table} ORDER BY id DESC LIMIT 200";
    if ($res = $conn->query($sql)) {
      while ($r = $res->fetch_assoc()) {
        echo '<tr data-type="'.$type.'" data-id="'.(int)$r['id']
           . '" data-open-digit="'.htmlspecialchars((string)$r['w_open'])
           . '" data-close-digit="'.htmlspecialchars((string)$r['w_close']).'">'
           . '<td>'.(int)$r['id'].'</td>'
           . '<td>'.(int)$r['user_id'].'</td>'
           . '<td>'.(int)$r['game_id'].'</td>'
           . '<td>'.htmlspecialchars($r['bid_details']).'</td>'
           . '<td>'.htmlspecialchars($r['total_bid_amount']).'</td>'
           . '<td>'.htmlspecialchars($r['created_at']).'</td>'
           . '<td>'.htmlspecialchars($r['game_name']).'</td>'
           . '<td>'.htmlspecialchars($r['bid_session']).'</td>'
           . '<td>'.htmlspecialchars($r['w_open']).'</td>'
           . '<td>'.htmlspecialchars($r['w_close']).'</td>'
           . '</tr>';
      }
      exit;
    }
  }
  echo '<tr><td colspan="10">No cache yet. Wait a minute for cron to build it.</td></tr>';
  exit;
}
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta http-equiv="Cache-Control" content="no-store, no-cache, must-revalidate, max-age=0">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Live — SD / DP</title>
<style>
  body { font-family: system-ui, Arial, sans-serif; margin:16px }
  .bar { display:flex; gap:8px; align-items:center; margin:4px 0 10px }
  .tab { padding:8px 12px; border:1px solid #ddd; background:#f7f7f7; cursor:pointer; border-radius:6px }
  .tab.active { background:#fff; border-color:#bbb; font-weight:700 }
  .btn { padding:8px 12px; border:1px solid #ddd; border-radius:6px; cursor:pointer; background:#f7f7f7 }
  .muted { color:#666; font-size:12px; margin:6px 0 10px }
  table { border-collapse: collapse; width:100%; font-size:14px }
  th,td { border:1px solid #e3e3e3; padding:6px 8px; text-align:left }
  th { background:#f7f7f7; position: sticky; top:0 }
  tbody tr:nth-child(even){ background:#fafafa }
</style>
</head>
<body>
  <h1>Live — Single Digit / Double Panna</h1>

  <div class="bar">
    <button class="tab active" data-type="sd">Single Digit</button>
    <button class="tab" data-type="dp">Double Panna</button>

    <!-- Sound controls -->
    <button id="soundToggle" class="btn" style="margin-left:auto">🔕 Sound: OFF</button>
    <label class="muted" style="display:flex;align-items:center;gap:6px">
      Vol <input id="vol" type="range" min="0" max="1" step="0.01" value="0.5" style="width:120px">
    </label>
  </div>

  <div class="muted" id="status">Loading…</div>

  <h2 id="latest" style="font:700 22px system-ui;margin:8px 0">
    <span id="label">SD</span> Latest — Open: <span id="openDigit">–</span> | Close: <span id="closeDigit">–</span>
  </h2>

  <table>
    <thead id="thead"></thead>
    <tbody id="rows"><tr><td>Loading…</td></tr></tbody>
  </table>

<script>
// ---------- UI refs ----------
const rowsEl   = document.getElementById('rows');
const statusEl = document.getElementById('status');
const headEl   = document.getElementById('thead');
const openEl   = document.getElementById('openDigit');
const closeEl  = document.getElementById('closeDigit');
const labelEl  = document.getElementById('label');
const soundBtn = document.getElementById('soundToggle');
const volEl    = document.getElementById('vol');

let currentType = 'sd';
let lastKey = '';

// ---------- table heads ----------
const heads = {
  sd: `<tr>
        <th>id</th><th>user_id</th><th>game_id</th><th>bid_details</th>
        <th>total_bid_amount</th><th>created_at</th><th>game_name</th>
        <th>bid_session</th><th>winning_open_time_sd</th><th>winning_close_time_sd</th>
      </tr>`,
  dp: `<tr>
        <th>id</th><th>user_id</th><th>game_id</th><th>bid_details</th>
        <th>total_bid_amount</th><th>created_at</th><th>game_name</th>
        <th>bid_session</th><th>winning_open_time_dp</th><th>winning_close_time_dp</th>
      </tr>`
};
function setHead(){ headEl.innerHTML = heads[currentType]; }
function setLabel(){ labelEl.textContent = currentType.toUpperCase(); }

// ---------- notifications ----------
async function askNotifPermission(){
  if (!('Notification' in window)) return;
  if (Notification.permission === 'default') { try { await Notification.requestPermission(); } catch(_){} }
}
function notifyChange(tag, od, cd, id) {
  if (!('Notification' in window)) return;
  if (Notification.permission === 'granted') {
    new Notification(`${tag} updated`, {
      body: `Row #${id} — Open: ${od ?? '—'} | Close: ${cd ?? '—'}`,
      icon: '/favicon.ico',
      tag: `live-${tag}`
    });
  }
  document.title = `${tag}: (${od ?? '•'}/${cd ?? '•'})`;
}

// ---------- sound (Web Audio) ----------
let soundEnabled = false;
let audioCtx = null, mainGain = null;

function initAudio() {
  if (!audioCtx) {
    const Ctx = window.AudioContext || window.webkitAudioContext;
    audioCtx = new Ctx();
    mainGain = audioCtx.createGain();
    mainGain.gain.value = +volEl.value || 0.5;
    mainGain.connect(audioCtx.destination);
  }
}
async function toggleSound() {
  initAudio();
  if (audioCtx.state === 'suspended') { try { await audioCtx.resume(); } catch(e){} }
  soundEnabled = !soundEnabled;
  soundBtn.textContent = soundEnabled ? '🔔 Sound: ON' : '🔕 Sound: OFF';
}
soundBtn.addEventListener('click', toggleSound);
volEl.addEventListener('input', () => { if (mainGain) mainGain.gain.value = +volEl.value; });

function beep(freq=880, ms=140) {
  if (!soundEnabled || !audioCtx) return;
  const osc = audioCtx.createOscillator();
  const g   = audioCtx.createGain();
  osc.type = 'sine';
  osc.frequency.value = freq;
  osc.connect(g); g.connect(mainGain);
  const t = audioCtx.currentTime;
  g.gain.setValueAtTime(0.0001, t);
  g.gain.linearRampToValueAtTime(0.9, t + 0.01);
  g.gain.exponentialRampToValueAtTime(0.0001, t + (ms/1000));
  osc.start(t);
  osc.stop(t + (ms/1000) + 0.02);
}
function playAlert(kind='sd') {
  if (!soundEnabled) return;
  if (kind === 'dp') {           // DP: triple descending
    beep(700,130); setTimeout(()=>beep(520,130),180); setTimeout(()=>beep(420,160),360);
  } else {                       // SD: double beep
    beep(900,120); setTimeout(()=>beep(660,120),180);
  }
}

// ---------- data refresh ----------
async function refreshNow() {
  try {
    const r = await fetch(`trys.php?ajax=1&type=${currentType}`, { cache:'no-store' });
    const html = await r.text();
    rowsEl.innerHTML = html || '<tr><td colspan="10">No rows</td></tr>';
    statusEl.textContent = 'Last update: ' + new Date().toLocaleTimeString();

    const first = rowsEl.querySelector('tr');
    if (first) {
      const id = +(first.dataset.id || 0);
      const od = first.dataset.openDigit ?? null;
      const cd = first.dataset.closeDigit ?? null;
      openEl.textContent  = (od === '' || od === 'null') ? '–' : od;
      closeEl.textContent = (cd === '' || cd === 'null') ? '–' : cd;

      const key = `${currentType}:${id}:${od}:${cd}`;
      if (key !== lastKey) {
        if (document.hidden) notifyChange(currentType.toUpperCase(), od, cd, id);
        playAlert(currentType);
        lastKey = key;
      }
    } else {
      openEl.textContent = closeEl.textContent = '–';
    }
  } catch(e) {
    statusEl.textContent = 'Error updating. Will retry…';
    console.error(e);
  }
}

// ---------- tab switching ----------
document.querySelectorAll('.tab').forEach(btn=>{
  btn.addEventListener('click', ()=>{
    document.querySelectorAll('.tab').forEach(b=>b.classList.remove('active'));
    btn.classList.add('active');
    currentType = btn.dataset.type;
    setHead(); setLabel();
    lastKey = '';         // force change detect on switch
    refreshNow();
  });
});

// ---------- boot ----------
askNotifPermission();     // requires HTTPS
setHead(); setLabel();
refreshNow();
setInterval(refreshNow, 2000); // UI refresh every 2s; cron keeps cache fresh ~5s
</script>
</body>
</html>
