<?php
// userdata.php
session_start();
require_once __DIR__ . '/db.php';  // must define $conn = new mysqli(...)

//
// 1) Auth guard
//
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}
$currentUserId = (int)$_SESSION['user_id'];

//
// 2) Lookup your own access level
//
$stmt = $conn->prepare("SELECT access_level FROM users_data WHERE user_id = ?");
$stmt->bind_param('i', $currentUserId);
$stmt->execute();
$stmt->bind_result($currentAccess);
$stmt->fetch();
$stmt->close();

//
// 3) Permissions
//
$isAdmin      = ($currentAccess === 'full');
$canSeeMobile = in_array($currentAccess, ['full','partial'], true);
$canSeeEmail  = in_array($currentAccess, ['full','partial'], true);
$canSeePass   = ($currentAccess === 'full');

//
// 4) Handle Delete (only for admins)
//
if ($isAdmin && $_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['delete_user_id'])) {
    $toDelete = (int)$_POST['delete_user_id'];
    // never delete yourself
    if ($toDelete !== $currentUserId) {
        $del = $conn->prepare("DELETE FROM users_data WHERE user_id = ?");
        $del->bind_param('i', $toDelete);
        $del->execute();
        $del->close();
    }
    header('Location: userdata.php?deleted=1');
    exit;
}

//
// 5) Handle Access-level updates (only for admins)
//
if ($isAdmin && $_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['access'])) {
    foreach ($_POST['access'] as $uid => $level) {
        $u = (int)$uid;
        if ($u !== $currentUserId && in_array($level, ['full','partial','none'], true)) {
            $upd = $conn->prepare("UPDATE users_data SET access_level = ? WHERE user_id = ?");
            $upd->bind_param('si', $level, $u);
            $upd->execute();
            $upd->close();
        }
    }
    header('Location: userdata.php?updated=1');
    exit;
}

//
// 6) Fetch all users
//
$res = $conn->query("
  SELECT user_id, username, email, mobile, password, access_level
    FROM users_data
   ORDER BY user_id
");
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>User Data & Access Control</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link
    href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css"
    rel="stylesheet">
  <style>
    .pwd-toggle {
      position: absolute;
      right: 12px;
      top: 50%;
      transform: translateY(-50%);
      cursor: pointer;
      font-size: 1.1rem;
      color: #555;
    }
    .pwd-input { padding-right: 2.5rem; }
  </style>
</head>
<body class="bg-light">
  <div class="container my-5">
    <h1>User Data & Access Control</h1>

    <?php if (isset($_GET['updated'])): ?>
      <div class="alert alert-success">Access levels updated.</div>
    <?php endif; ?>
    <?php if (isset($_GET['deleted'])): ?>
      <div class="alert alert-success">User deleted.</div>
    <?php endif; ?>

    <form method="post">
      <div class="table-responsive">
        <table class="table table-striped table-bordered align-middle">
          <thead class="table-light">
            <tr>
              <th>User ID</th>
              <th>Username</th>
              <?php if ($canSeeMobile): ?><th>Mobile</th><?php endif; ?>
              <?php if ($canSeeEmail):  ?><th>Email</th><?php endif; ?>
              <?php if ($canSeePass):   ?><th>Password</th><?php endif; ?>
              <?php if ($isAdmin): ?>
                <th>Access Level</th>
                <th>Action</th>
              <?php endif; ?>
            </tr>
          </thead>
          <tbody>
            <?php while ($u = $res->fetch_assoc()): ?>
            <tr>
              <td><?= $u['user_id'] ?></td>
              <td><?= htmlspecialchars($u['username']) ?></td>

              <?php if ($canSeeMobile): ?>
                <td><?= htmlspecialchars($u['mobile']) ?></td>
              <?php endif; ?>

              <?php if ($canSeeEmail): ?>
                <td><?= htmlspecialchars($u['email']) ?></td>
              <?php endif; ?>

              <?php if ($canSeePass): 
                $raw = $u['password'];
                $masked = (strpos($raw,'$2y$')===0) ? '********' : $raw;
                $type   = (strpos($raw,'$2y$')===0) ? 'password' : 'text';
              ?>
                <td style="position:relative;">
                  <input
                    id="pwd-<?= $u['user_id'] ?>"
                    type="<?= $type ?>"
                    readonly
                    class="form-control pwd-input"
                    value="<?= htmlspecialchars($masked) ?>">
                  <span
                    class="pwd-toggle"
                    data-target="pwd-<?= $u['user_id'] ?>">👁️</span>
                </td>
              <?php endif; ?>

              <?php if ($isAdmin): ?>
                <td>
                  <div class="form-check form-check-inline">
                    <input class="form-check-input" type="radio"
                           name="access[<?= $u['user_id'] ?>]"
                           value="full"
                           id="full-<?= $u['user_id'] ?>"
                           <?= $u['access_level']==='full'?'checked':'' ?>>
                    <label class="form-check-label" for="full-<?= $u['user_id'] ?>">Full</label>
                  </div>
                  <div class="form-check form-check-inline">
                    <input class="form-check-input" type="radio"
                           name="access[<?= $u['user_id'] ?>]"
                           value="partial"
                           id="partial-<?= $u['user_id'] ?>"
                           <?= $u['access_level']==='partial'?'checked':'' ?>>
                    <label class="form-check-label" for="partial-<?= $u['user_id'] ?>">Partial</label>
                  </div>
                  <div class="form-check form-check-inline">
                    <input class="form-check-input" type="radio"
                           name="access[<?= $u['user_id'] ?>]"
                           value="none"
                           id="none-<?= $u['user_id'] ?>"
                           <?= $u['access_level']==='none'?'checked':'' ?>>
                    <label class="form-check-label" for="none-<?= $u['user_id'] ?>">None</label>
                  </div>
                </td>
                <td>
                  <?php if ($u['user_id'] != $currentUserId): ?>
                    <button
                      type="submit"
                      name="delete_user_id"
                      value="<?= $u['user_id'] ?>"
                      class="btn btn-sm btn-danger"
                      onclick="return confirm('Delete <?= htmlspecialchars($u['username']) ?>?')">
                      Delete
                    </button>
                  <?php else: ?>
                    &mdash;
                  <?php endif; ?>
                </td>
              <?php endif; ?>
            </tr>
            <?php endwhile; ?>
          </tbody>
        </table>
      </div>

      <?php if ($isAdmin): ?>
        <button type="submit" class="btn btn-success mt-2">Save Access Levels</button>
      <?php endif; ?>
    </form>
  </div>

  <script>
    // Toggle password visibility
    document.querySelectorAll('.pwd-toggle').forEach(el => {
      el.addEventListener('click', () => {
        const fld = document.getElementById(el.dataset.target);
        if (fld.type === 'password') {
          fld.type = 'text';
          el.textContent = '🙈';
        } else {
          fld.type = 'password';
          el.textContent = '👁️';
        }
      });
    });
  </script>
</body>
</html>
