<?php
session_start();

// Include the database connection file
include('db.php');

// Check if the user is logged in, if not, redirect to the login page
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$message = '';

// Handle Set Balance functionality (existing functionality renamed)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['set_balance'])) {
    $user_id     = (int)$_POST['user_id'];
    $new_balance = (float)$_POST['new_balance'];

    // Validate input
    if ($user_id <= 0) {
        $message = "Invalid user ID!";
    } elseif ($new_balance < 0) {
        $message = "Balance cannot be negative!";
    } else {
        // Check if wallet exists for this user
        $checkSql = "SELECT COUNT(*) AS count FROM wallet WHERE user_id = ?";
        $checkStmt = $conn->prepare($checkSql);
        $checkStmt->bind_param("i", $user_id);
        $checkStmt->execute();
        $checkStmt->bind_result($count);
        $checkStmt->fetch();
        $checkStmt->close();

        if ($count > 0) {
            // Update existing wallet
            $updateSql = "UPDATE wallet SET balance = ? WHERE user_id = ?";
            $updateStmt = $conn->prepare($updateSql);
            $updateStmt->bind_param("di", $new_balance, $user_id);
            
            if ($updateStmt->execute()) {
                $message = "Balance set to ₹" . number_format($new_balance, 2) . " successfully!";
            } else {
                $message = "Error updating balance: " . $conn->error;
            }
            $updateStmt->close();
        } else {
            // Create new wallet
            $insertSql = "INSERT INTO wallet (user_id, balance) VALUES (?, ?)";
            $insertStmt = $conn->prepare($insertSql);
            $insertStmt->bind_param("id", $user_id, $new_balance);
            
            if ($insertStmt->execute()) {
                $message = "Wallet created with balance ₹" . number_format($new_balance, 2) . " successfully!";
            } else {
                $message = "Error creating wallet: " . $conn->error;
            }
            $insertStmt->close();
        }
    }
}

// Handle Add Balance functionality (NEW - adds to existing balance)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_balance'])) {
    $user_id = (int)$_POST['user_id'];
    $add_amount = (float)$_POST['add_amount'];

    // Validate input
    if ($user_id <= 0) {
        $message = "Invalid user ID!";
    } elseif ($add_amount <= 0) {
        $message = "Amount to add must be greater than 0!";
    } else {
        // Get current balance
        $getCurrentSql = "SELECT balance FROM wallet WHERE user_id = ?";
        $getCurrentStmt = $conn->prepare($getCurrentSql);
        
        if ($getCurrentStmt) {
            $getCurrentStmt->bind_param("i", $user_id);
            $getCurrentStmt->execute();
            $getCurrentStmt->bind_result($current_balance);
            
            if ($getCurrentStmt->fetch()) {
                // User has existing wallet
                $getCurrentStmt->close();
                
                $new_balance = $current_balance + $add_amount;
                
                $updateSql = "UPDATE wallet SET balance = ? WHERE user_id = ?";
                $updateStmt = $conn->prepare($updateSql);
                $updateStmt->bind_param("di", $new_balance, $user_id);
                
                if ($updateStmt->execute()) {
                    $message = "₹" . number_format($add_amount, 2) . " added successfully! New balance: ₹" . number_format($new_balance, 2);
                } else {
                    $message = "Error adding balance: " . $conn->error;
                }
                $updateStmt->close();
            } else {
                // User doesn't have wallet, create one with the add amount
                $getCurrentStmt->close();
                
                $insertSql = "INSERT INTO wallet (user_id, balance) VALUES (?, ?)";
                $insertStmt = $conn->prepare($insertSql);
                $insertStmt->bind_param("id", $user_id, $add_amount);
                
                if ($insertStmt->execute()) {
                    $message = "Wallet created and ₹" . number_format($add_amount, 2) . " added successfully!";
                } else {
                    $message = "Error creating wallet: " . $conn->error;
                }
                $insertStmt->close();
            }
        } else {
            $message = "Database error: " . $conn->error;
        }
    }
}

// Handle Delete Wallet functionality
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_wallet'])) {
    $user_id = (int)$_POST['user_id'];
    
    if ($user_id <= 0) {
        $message = "Invalid user ID!";
    } else {
        $deleteSql = "DELETE FROM wallet WHERE user_id = ?";
        $deleteStmt = $conn->prepare($deleteSql);
        $deleteStmt->bind_param("i", $user_id);
        
        if ($deleteStmt->execute()) {
            if ($deleteStmt->affected_rows > 0) {
                $message = "Wallet deleted successfully!";
            } else {
                $message = "No wallet found to delete!";
            }
        } else {
            $message = "Error deleting wallet: " . $conn->error;
        }
        $deleteStmt->close();
    }
}

// Clean up duplicate wallet entries (run this once to fix existing duplicates)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cleanup_duplicates'])) {
    // Keep only the latest wallet entry for each user_id
    $cleanupSql = "DELETE w1 FROM wallet w1 
                   INNER JOIN wallet w2 
                   WHERE w1.user_id = w2.user_id 
                   AND w1.id < w2.id";
    
    if ($conn->query($cleanupSql)) {
        $message = "Duplicate wallet entries cleaned up successfully!";
    } else {
        $message = "Error cleaning up duplicates: " . $conn->error;
    }
}

// Fetch all users and their wallet balances (LEFT JOIN to show all users, even those without wallets)
$sql = "SELECT u.user_id, u.username, COALESCE(w.balance, 0.00) as balance,
               CASE WHEN w.user_id IS NOT NULL THEN 1 ELSE 0 END as has_wallet
        FROM users_data u 
        LEFT JOIN wallet w ON u.user_id = w.user_id
        ORDER BY u.username ASC";

$result = $conn->query($sql);
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>User Wallet Information</title>
  <style>
    body { 
      font-family: Arial, sans-serif; 
      background-color: #f0f2f5; 
      margin: 0; 
      padding: 0; 
    }
    
    .navbar { 
      background-color: #007bff; 
      padding: 15px; 
      color: white; 
      text-align: center;
      position: relative;
    }
    
    .back-button { 
      position: absolute; 
      top: 15px; 
      left: 15px; 
      background-color: #28a745;
      color: white; 
      padding: 8px 12px; 
      text-decoration: none; 
      border-radius: 5px;
      font-size: 14px;
    }
    
    .back-button:hover { 
      background-color: #218838; 
    }
    
    .container { 
      padding: 20px; 
      max-width: 1000px; 
      margin: 20px auto; 
      background-color: #fff; 
      border-radius: 8px;
      box-shadow: 0 0 10px rgba(0,0,0,0.1); 
    }
    
    .cleanup-section {
      background-color: #fff3cd;
      border: 1px solid #ffeaa7;
      padding: 15px;
      margin-bottom: 20px;
      border-radius: 5px;
    }
    
    .table { 
      width: 100%; 
      margin-top: 20px; 
      border-collapse: collapse;
      font-size: 14px;
    }
    
    .table th, .table td { 
      padding: 12px 8px; 
      text-align: left; 
      border: 1px solid #ddd;
      vertical-align: middle;
    }
    
    .table th { 
      background-color: #007bff; 
      color: white;
      font-weight: bold;
    }
    
    .table tr:nth-child(even) {
      background-color: #f8f9fa;
    }
    
    .table tr:hover {
      background-color: #e9ecef;
    }
    
    .form-group { 
      display: flex;
      flex-direction: column;
      gap: 5px;
      margin: 2px 0;
    }
    
    .form-row {
      display: flex;
      align-items: center;
      gap: 5px;
      flex-wrap: wrap;
    }
    
    .form-group input[type="number"] { 
      padding: 6px 8px; 
      width: 100px; 
      font-size: 13px;
      border: 1px solid #ddd;
      border-radius: 4px;
    }
    
    .form-group input[type="number"].add-input { 
      border-color: #17a2b8;
      background-color: #f0f9ff;
    }
    
    .button { 
      padding: 6px 12px; 
      color: white; 
      border: none;
      border-radius: 4px; 
      cursor: pointer;
      font-size: 12px;
      margin: 1px;
    }
    
    .button.set {
      background-color: #28a745;
    }
    
    .button.set:hover { 
      background-color: #218838; 
    }
    
    .button.add {
      background-color: #17a2b8;
    }
    
    .button.add:hover { 
      background-color: #138496; 
    }
    
    .button.delete { 
      background-color: #dc3545; 
    }
    
    .button.delete:hover { 
      background-color: #c82333; 
    }
    
    .button.cleanup {
      background-color: #ffc107;
      color: #212529;
    }
    
    .button.cleanup:hover {
      background-color: #e0a800;
    }
    
    .message { 
      margin: 15px 0; 
      padding: 10px; 
      border-radius: 4px;
      font-weight: bold;
    }
    
    .message.success {
      background-color: #d4edda;
      border: 1px solid #c3e6cb;
      color: #155724;
    }
    
    .message.error {
      background-color: #f8d7da;
      border: 1px solid #f5c6cb;
      color: #721c24;
    }
    
    .no-wallet {
      color: #6c757d;
      font-style: italic;
    }
    
    .stats {
      display: flex;
      justify-content: space-around;
      background-color: #e9ecef;
      padding: 15px;
      border-radius: 5px;
      margin-bottom: 20px;
    }
    
    .stat-item {
      text-align: center;
    }
    
    .stat-number {
      font-size: 24px;
      font-weight: bold;
      color: #007bff;
    }
    
    .stat-label {
      font-size: 12px;
      color: #6c757d;
    }
    
    .action-label {
      font-size: 11px;
      color: #6c757d;
      margin-bottom: 2px;
    }
  </style>
</head>
<body>

  <div class="navbar">
    <a href="dashboard.php" class="back-button">← Back</a>
    <h2>User Wallet Management</h2>
  </div>

  <div class="container">
    
    <?php if (!empty($message)): ?>
      <div class="message <?= strpos($message, 'Error') !== false ? 'error' : 'success' ?>">
        <?= htmlspecialchars($message) ?>
      </div>
    <?php endif; ?>

    <?php
    // Calculate statistics
    $totalUsers = 0;
    $usersWithWallets = 0;
    $totalBalance = 0;
    
    if ($result && $result->num_rows > 0) {
      $result->data_seek(0); // Reset pointer
      while ($row = $result->fetch_assoc()) {
        $totalUsers++;
        if ($row['has_wallet']) {
          $usersWithWallets++;
          $totalBalance += $row['balance'];
        }
      }
      $result->data_seek(0); // Reset pointer again
    }
    ?>
    
    <div class="stats">
      <div class="stat-item">
        <div class="stat-number"><?= $totalUsers ?></div>
        <div class="stat-label">Total Users</div>
      </div>
      <div class="stat-item">
        <div class="stat-number"><?= $usersWithWallets ?></div>
        <div class="stat-label">Users with Wallets</div>
      </div>
      <div class="stat-item">
        <div class="stat-number">₹<?= number_format($totalBalance, 2) ?></div>
        <div class="stat-label">Total Balance</div>
      </div>
    </div>

    <div class="cleanup-section">
      <h4>Database Maintenance</h4>
      <p>If you see duplicate users, click below to remove duplicate wallet entries:</p>
      <form method="POST" style="display:inline;">
        <button type="submit" name="cleanup_duplicates" class="button cleanup" 
                onclick="return confirm('Are you sure you want to clean up duplicate wallet entries?')">
          Clean Up Duplicates
        </button>
      </form>
    </div>

    <h3>Users and Their Wallet Balances</h3>

    <table class="table">
      <thead>
        <tr>
          <th>User ID</th>
          <th>Username</th>
          <th>Current Balance (₹)</th>
          <th>Status</th>
          <th>Actions</th>
        </tr>
      </thead>
      <tbody>
        <?php if ($result && $result->num_rows > 0): ?>
          <?php while ($row = $result->fetch_assoc()): ?>
            <tr>
              <td><?= (int)$row['user_id'] ?></td>
              <td><?= htmlspecialchars($row['username']) ?></td>
              <td>
                <?php if ($row['has_wallet']): ?>
                  ₹<?= number_format($row['balance'], 2) ?>
                <?php else: ?>
                  <span class="no-wallet">No wallet</span>
                <?php endif; ?>
              </td>
              <td>
                <?php if ($row['has_wallet']): ?>
                  <span style="color: #28a745;">✓ Active</span>
                <?php else: ?>
                  <span style="color: #6c757d;">○ No wallet</span>
                <?php endif; ?>
              </td>
              <td>
                <div class="form-group">
                  <!-- Set Balance Form -->
                  <div class="action-label">Set Total Balance:</div>
                  <div class="form-row">
                    <form method="POST" style="display:inline-flex; align-items:center; gap:3px;">
                      <input type="hidden" name="user_id" value="<?= (int)$row['user_id'] ?>">
                      <input type="number" 
                             name="new_balance" 
                             value="<?= $row['has_wallet'] ? number_format($row['balance'], 2, '.', '') : '0.00' ?>" 
                             step="0.01" 
                             min="0"
                             placeholder="Total balance"
                             title="Set total balance to this amount"
                             required>
                      <button type="submit" name="set_balance" class="button set">
                        <?= $row['has_wallet'] ? 'Set' : 'Create' ?>
                      </button>
                    </form>
                  </div>
                  
                  <?php if ($row['has_wallet']): ?>
                    <!-- Add Balance Form -->
                    <div class="action-label">Add to Current Balance:</div>
                    <div class="form-row">
                      <form method="POST" style="display:inline-flex; align-items:center; gap:3px;">
                        <input type="hidden" name="user_id" value="<?= (int)$row['user_id'] ?>">
                        <input type="number" 
                               name="add_amount" 
                               step="0.01" 
                               min="0.01"
                               placeholder="Amount to add"
                               title="Amount to add to current balance (<?= number_format($row['balance'], 2) ?>)"
                               class="add-input"
                               required>
                        <button type="submit" name="add_balance" class="button add">
                          + Add
                        </button>
                      </form>
                      
                      <!-- Delete Wallet Button -->
                      <form method="POST" style="display:inline;">
                        <input type="hidden" name="user_id" value="<?= (int)$row['user_id'] ?>">
                        <button type="submit" 
                                name="delete_wallet" 
                                class="button delete"
                                onclick="return confirm('Are you sure you want to delete this wallet?')">
                          Delete
                        </button>
                      </form>
                    </div>
                  <?php endif; ?>
                </div>
              </td>
            </tr>
          <?php endwhile; ?>
        <?php else: ?>
          <tr><td colspan="5" style="text-align:center; padding:20px;">No users found</td></tr>
        <?php endif; ?>
      </tbody>
    </table>
    
    <div style="margin-top: 20px; padding: 15px; background-color: #f8f9fa; border-radius: 5px; font-size: 13px; color: #6c757d;">
      <strong>Instructions:</strong><br>
      • <strong>Set Total Balance:</strong> Enter the exact amount you want the balance to be<br>
      • <strong>Add to Balance:</strong> Enter an amount to add to the current balance (only for existing wallets)<br>
      • <strong>Example:</strong> If Sam has ₹200 and you add ₹100, the new balance will be ₹300
    </div>
  </div>

</body>
</html>
<?php
$conn->close();
?>