<?php
// Show errors during dev
ini_set('display_errors',1);
ini_set('display_startup_errors',1);
error_reporting(E_ALL);

session_start();
include __DIR__ . '/db.php';

// (Optional) require login
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

// 1) Validate game_id
if (empty($_GET['game_id']) || !ctype_digit($_GET['game_id'])) {
    header('Location: games.php');
    exit;
}
$game_id = (int)$_GET['game_id'];

// 2) Fetch game metadata
$stmt = $conn->prepare("
  SELECT game_name, sub_games, winning_number
    FROM game56
   WHERE id = ?
");
$stmt->bind_param("i", $game_id);
$stmt->execute();
$stmt->bind_result($game_name, $subname, $winning_number);
if (!$stmt->fetch()) {
    die("Game not found.");
}
$stmt->close();

// 3) Fetch all bids + usernames
$q = $conn->prepare("
  SELECT ub.bid_details, u.username
    FROM user_bids AS ub
    JOIN users_data AS u
      ON ub.user_id = u.user_id
   WHERE ub.game_id = ?
");
$q->bind_param("i", $game_id);
$q->execute();
$q->store_result();
$q->bind_result($bid_json, $username);

// 4) Build result rows
$rows = [];
while ($q->fetch()) {
    $bids = json_decode($bid_json, true) ?: [];
    // Win if they staked on the winning_number
    $isWin = isset($bids[$winning_number]) && $bids[$winning_number] > 0;
    $rows[] = [
      'username' => $username,
      'bids'     => $bids,
      'result'   => $isWin ? 'Win' : 'Lose'
    ];
}
$q->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Game Result</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body { font-family:Arial,sans-serif; background:#f4f4f9; color:#333; margin:0; padding:0 }
    .container { max-width:800px; margin:40px auto; background:#fff; padding:20px; 
                 border-radius:8px; box-shadow:0 2px 5px rgba(0,0,0,0.1) }
    h1 { text-align:center; color:#1E90FF }
    .info { margin:10px 0; font-size:1rem }
    .info strong { display:inline-block; width:140px }
    table { width:100%; border-collapse:collapse; margin-top:20px }
    th, td { border:1px solid #ddd; padding:8px; vertical-align:top }
    th { background:#007BFF; color:#fff; }
    .bid-list { font-size:0.9rem; }
    .bid-list div { margin-bottom:4px; }
    .win { color:green; font-weight:bold; }
    .lose { color:red; font-weight:bold; }
    .back { display:inline-block; margin-bottom:20px; color:#1E90FF; text-decoration:none }
    .back:hover { text-decoration:underline }
  </style>
</head>
<body>
  <div class="container">
    <a href="games.php" class="back">← Back to Games</a>
    <h1>Result for “<?= htmlspecialchars($game_name) ?>”</h1>
    <p class="info"><strong>Sub‑game:</strong> <?= htmlspecialchars($subname) ?></p>
    <p class="info"><strong>Winning Number:</strong> <?= $winning_number ?></p>

    <table>
      <tr>
        <th>User Name</th>
        <th>Bid Details</th>
        <th>Result</th>
      </tr>
      <?php if (empty($rows)): ?>
        <tr><td colspan="3" style="text-align:center; padding:20px;">No bids placed.</td></tr>
      <?php else: ?>
        <?php foreach ($rows as $r): ?>
          <tr>
            <td><?= htmlspecialchars($r['username']) ?></td>
            <td class="bid-list">
              <?php if (empty($r['bids'])): ?>
                <em>No bids</em>
              <?php else: ?>
                <?php foreach ($r['bids'] as $digit => $amt): ?>
                  <div><?= $digit ?>: ₹<?= number_format($amt,2) ?></div>
                <?php endforeach; ?>
              <?php endif; ?>
            </td>
            <td class="<?= strtolower($r['result']) ?>">
              <?= $r['result'] ?>
            </td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </table>
  </div>
</body>
</html>
